/* FILE: mfreerun.c                             (D. Tottingham  12/26/90)

This is a collection of C functions that manage the free run events for xdetect.
All functions have been written and compiled medium model.  The following
functions are included:

fr_continue_freerun ()          continue free run event if started
fr_get_block_size ()            get block size in seconds
fr_initialize ()                initialize free run parameters
fr_set_block_size ()            set block size in seconds
fr_start_freerun ()             start free run event
fr_stop_freerun ()              stop free run event
fr_toggle_freerun ()            toggle free run event


EXTERNAL FUNCTIONS CALLED:

dm_get_current_time ()          get current time
f_write_buffers ()              write data buffer(s) to disk
h_toggle_status ()              toggle a status bit

HISTORY:
   none

*/


/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <stdio.h>

#include "mconst.h"
#include "mdemux.h"
#include "mfile.h"
#include "mfreerun.h"
#include "mscrnhdr.h"


/*************************************************************************
                                GLOBALS

*************************************************************************/
PRIVATE unsigned int freerun_state;
PRIVATE double block_size, next_block_time;


/*=======================================================================*
 *                          fr_continue_freerun                          *
 *=======================================================================*/
/* Continue free run event if started.                                   */

PUBLIC
FLAG fr_continue_freerun ()
{
   switch (freerun_state) {
      case START_FREERUN:
         f_write_buffers (START_FILE, F_FREERUN);
         next_block_time = dm_get_current_time() + block_size;
         freerun_state = CONTINUE_FREERUN;
         break;

      case CONTINUE_FREERUN:
         if (block_size == 0.0 || dm_get_current_time() < next_block_time)
            f_write_buffers (CONTINUE_FILE, F_FREERUN);
         else {
            f_write_buffers (END_FILE, F_FREERUN);
            f_write_buffers (START_FILE, F_FREERUN);
            next_block_time = dm_get_current_time() + block_size;
         }
         break;

      case STOP_FREERUN:
         f_write_buffers (END_FILE, F_FREERUN);
         freerun_state = READY_STATE;
         break;

      case READY_STATE:
      default:
         break;
   }
   return (freerun_state == CONTINUE_FREERUN);
}

/*=======================================================================*
 *                           fr_get_block_size                           *
 *=======================================================================*/
/* Get block size in seconds.                                            */

PUBLIC
double fr_get_block_size ()
{
   return block_size;
}

/*=======================================================================*
 *                             fr_initialize                             *
 *=======================================================================*/
/* Initialize free run parameters.                                       */

PUBLIC
void fr_initialize ()
{
   freerun_state = READY_STATE;
   block_size = next_block_time = 0.0;
}

/*=======================================================================*
 *                           fr_set_block_size                           *
 *=======================================================================*/
/* Set block size in seconds.                                            */

PUBLIC
void fr_set_block_size (size)
double size;
{
   block_size = size;
}

/*=======================================================================*
 *                            fr_start_freerun                           *
 *=======================================================================*/
/* Start free run event.                                                 */

PUBLIC
void fr_start_freerun ()
{
   if (freerun_state == READY_STATE) {
      freerun_state = START_FREERUN;
      h_toggle_status (H_FREE_RUN);
   }
}

/*=======================================================================*
 *                             fr_stop_freerun                           *
 *=======================================================================*/
/* Stop free run event.                                                  */

PUBLIC
void fr_stop_freerun ()
{
   if (freerun_state == CONTINUE_FREERUN) {
      freerun_state = STOP_FREERUN;
      h_toggle_status (H_FREE_RUN);
   }
}

/*=======================================================================*
 *                            fr_toggle_freerun                          *
 *=======================================================================*/
/* Toggle free run event.                                                */

PUBLIC
void fr_toggle_freerun ()
{
   if (freerun_state == READY_STATE) {
      freerun_state = START_FREERUN;
      h_toggle_status (H_FREE_RUN);
   } else if (freerun_state == CONTINUE_FREERUN) {
      freerun_state = STOP_FREERUN;
      h_toggle_status (H_FREE_RUN);
   }
}
